<?php

use App\Http\Controllers\Auth\ForgotPasswordController;
use App\Http\Controllers\Auth\ResetPasswordController;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\FlashSaleController;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\OrderController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\ProfileController;
use Illuminate\Support\Facades\Route;

// Redirect root ke /id
Route::get('/', function () {
    return redirect('/id');
});

// Semua routes dengan prefix /id
Route::prefix('id')->group(function () {
    // Homepage - Game Store
    Route::get('/', [HomeController::class, 'index'])->name('home');

    // Flash Sale Routes
    Route::get('/flash-sale', [FlashSaleController::class, 'index'])->name('flash-sale');
    Route::get('/flash-sale/{flashSale}/status', [FlashSaleController::class, 'status'])->name('flash-sale.status');

    // Category Routes
    Route::get('/category/{slug}', [HomeController::class, 'category'])->name('category.show');

    // Product Routes (Game specific products)
    Route::get('/products/{game:slug}', [ProductController::class, 'index'])->name('products.show');
    Route::post('/products/{game:slug}/checkout', [ProductController::class, 'checkout'])->name('products.checkout');

    // Order Routes (tanpa login)
    Route::get('/order/{orderId}', [OrderController::class, 'checkout'])->name('order.checkout');
    Route::post('/order/{orderId}/confirm', [OrderController::class, 'confirmPayment'])->name('order.confirm');
    Route::get('/order/{orderId}/status', [OrderController::class, 'status'])->name('order.status');
    Route::get('/track', [OrderController::class, 'track'])->name('order.track');
    Route::get('/api/order/{orderId}/status', [OrderController::class, 'getStatus'])->name('order.api.status');

    // Authentication Routes
    Route::get('/login', [AuthController::class, 'showLogin'])->name('login');
    Route::post('/login', [AuthController::class, 'login']);
    Route::get('/register', [AuthController::class, 'showRegister'])->name('register');
    Route::post('/register', [AuthController::class, 'register']);
    Route::post('/logout', [AuthController::class, 'logout'])->name('logout');

    // Email Verification Code Routes (untuk register)
    Route::post('/send-verification-code', [AuthController::class, 'sendVerificationCode'])->name('send.verification.code');
    Route::post('/verify-code', [AuthController::class, 'verifyCode'])->name('verify.code');

    // Password Reset Routes
    Route::get('/forgot-password', [ForgotPasswordController::class, 'showLinkRequestForm'])->name('password.request');
    Route::post('/forgot-password', [ForgotPasswordController::class, 'sendResetLinkEmail'])->name('password.email');
    Route::get('/reset-password/{token}', [ResetPasswordController::class, 'showResetForm'])->name('password.reset');
    Route::post('/reset-password', [ResetPasswordController::class, 'reset'])->name('password.update');

    // Email Verification Routes
    Route::get('/email/verify', function () {
        return view('auth.verify-email');
    })->middleware('auth')->name('verification.notice');

    Route::get('/email/verify/{id}/{hash}', function (\Illuminate\Foundation\Auth\EmailVerificationRequest $request) {
        $request->fulfill();
        return redirect('/')->with('success', 'Email berhasil diverifikasi!');
    })->middleware(['auth', 'signed'])->name('verification.verify');

    Route::post('/email/verification-notification', function (\Illuminate\Http\Request $request) {
        $request->user()->sendEmailVerificationNotification();
        return back()->with('status', 'Link verifikasi telah dikirim ke email Anda!');
    })->middleware(['auth', 'throttle:6,1'])->name('verification.send');

    // Terms & Conditions
    Route::get('/terms', function () {
        return view('terms');
    })->name('terms');

    // Protected Routes (require login and email verification)
    Route::middleware(['auth', 'verified'])->group(function () {
        // Profile Routes
        Route::get('/profile', [ProfileController::class, 'index'])->name('profile');
        Route::put('/profile', [ProfileController::class, 'update'])->name('profile.update');
        Route::get('/profile/change-password', [ProfileController::class, 'showChangePasswordForm'])->name('profile.change-password');
        Route::put('/profile/password', [ProfileController::class, 'updatePassword'])->name('profile.password');
    });

    // Admin Routes
    Route::prefix('admin')->name('admin.')->group(function () {
        // Admin Login (no middleware)
        Route::get('/login', [\App\Http\Controllers\Admin\AdminAuthController::class, 'showLogin'])->name('login');
        Route::post('/login', [\App\Http\Controllers\Admin\AdminAuthController::class, 'login'])->name('login.post');
        Route::post('/logout', [\App\Http\Controllers\Admin\AdminAuthController::class, 'logout'])->name('logout');

        // Admin Panel (requires admin middleware)
        Route::middleware(['admin'])->group(function () {
            Route::get('/', [\App\Http\Controllers\Admin\AdminController::class, 'dashboard'])->name('dashboard');

            // Category Management
            Route::resource('categories', \App\Http\Controllers\Admin\CategoryController::class);

            // Game Management
            Route::resource('games', \App\Http\Controllers\Admin\GameController::class);

            // Product Management
            Route::resource('products', \App\Http\Controllers\Admin\ProductController::class);

            // Flash Sale Management
            Route::resource('flash-sales', \App\Http\Controllers\Admin\FlashSaleController::class);

            // Order Management
            Route::get('orders', [\App\Http\Controllers\Admin\OrderController::class, 'index'])->name('orders.index');
            Route::get('orders/{order}', [\App\Http\Controllers\Admin\OrderController::class, 'show'])->name('orders.show');
            Route::patch('orders/{order}/status', [\App\Http\Controllers\Admin\OrderController::class, 'updateStatus'])->name('orders.update-status');
            Route::post('orders/bulk-update', [\App\Http\Controllers\Admin\OrderController::class, 'bulkUpdate'])->name('orders.bulk-update');
            Route::delete('orders/{order}', [\App\Http\Controllers\Admin\OrderController::class, 'destroy'])->name('orders.destroy');
        });
    });
});
