@extends('layouts.admin')

@section('title', 'Kelola Pesanan')

@section('content')
<div class="admin-header">
    <h1>Kelola Pesanan</h1>
</div>

@if(session('success'))
    <div class="alert alert-success">
        {{ session('success') }}
    </div>
@endif

<!-- Filter & Search -->
<div class="order-filters">
    <div class="filter-tabs">
        <a href="{{ route('admin.orders.index', ['status' => 'all']) }}" 
           class="filter-tab {{ request('status', 'all') === 'all' ? 'active' : '' }}">
            Semua <span class="badge">{{ $counts['all'] }}</span>
        </a>
        <a href="{{ route('admin.orders.index', ['status' => 'pending']) }}" 
           class="filter-tab {{ request('status') === 'pending' ? 'active' : '' }}">
            Pending <span class="badge warning">{{ $counts['pending'] }}</span>
        </a>
        <a href="{{ route('admin.orders.index', ['status' => 'paid']) }}" 
           class="filter-tab {{ request('status') === 'paid' ? 'active' : '' }}">
            Dibayar <span class="badge info">{{ $counts['paid'] }}</span>
        </a>
        <a href="{{ route('admin.orders.index', ['status' => 'processing']) }}" 
           class="filter-tab {{ request('status') === 'processing' ? 'active' : '' }}">
            Diproses <span class="badge">{{ $counts['processing'] }}</span>
        </a>
        <a href="{{ route('admin.orders.index', ['status' => 'completed']) }}" 
           class="filter-tab {{ request('status') === 'completed' ? 'active' : '' }}">
            Selesai <span class="badge success">{{ $counts['completed'] }}</span>
        </a>
        <a href="{{ route('admin.orders.index', ['status' => 'failed']) }}" 
           class="filter-tab {{ request('status') === 'failed' ? 'active' : '' }}">
            Gagal <span class="badge danger">{{ $counts['failed'] }}</span>
        </a>
    </div>
    
    <form action="{{ route('admin.orders.index') }}" method="GET" class="search-form">
        <input type="hidden" name="status" value="{{ request('status', 'all') }}">
        <input type="text" name="search" placeholder="Cari Order ID, WhatsApp, atau Game ID..." 
               value="{{ request('search') }}" class="search-input">
        <button type="submit" class="btn btn-primary">
            <i class="fas fa-search"></i> Cari
        </button>
    </form>
</div>

<!-- Bulk Actions -->
<form id="bulkForm" action="{{ route('admin.orders.bulk-update') }}" method="POST">
    @csrf
    <div class="bulk-actions" style="display: none;">
        <span class="selected-count">0 pesanan dipilih</span>
        <select name="action" class="bulk-action-select">
            <option value="">Pilih Aksi</option>
            <option value="mark_paid">Tandai Dibayar</option>
            <option value="mark_processing">Tandai Diproses</option>
            <option value="mark_completed">Tandai Selesai</option>
            <option value="mark_failed">Tandai Gagal</option>
        </select>
        <button type="submit" class="btn btn-secondary">Terapkan</button>
    </div>

    <!-- Orders Table -->
    <div class="admin-table-container">
        <table class="admin-table">
            <thead>
                <tr>
                    <th width="40">
                        <input type="checkbox" id="selectAll" class="checkbox">
                    </th>
                    <th>Order ID</th>
                    <th>Produk</th>
                    <th>ID Game</th>
                    <th>WhatsApp</th>
                    <th>Total</th>
                    <th>Status</th>
                    <th>Tanggal</th>
                    <th>Aksi</th>
                </tr>
            </thead>
            <tbody>
                @forelse($orders as $order)
                <tr>
                    <td>
                        <input type="checkbox" name="order_ids[]" value="{{ $order->id }}" class="checkbox order-checkbox">
                    </td>
                    <td>
                        <strong class="order-id">{{ $order->order_id }}</strong>
                        @if($order->flash_sale_id)
                            <span class="flash-badge" title="Flash Sale">⚡</span>
                        @endif
                    </td>
                    <td>
                        <div class="product-info">
                            @if($order->product)
                                <span class="product-name">{{ $order->product->name }}</span>
                                <span class="game-name">{{ $order->product->game->name ?? '-' }}</span>
                            @else
                                <span class="text-muted">Produk dihapus</span>
                            @endif
                        </div>
                    </td>
                    <td>{{ $order->game_user_id }}</td>
                    <td>{{ $order->whatsapp }}</td>
                    <td>
                        <div class="price-info">
                            <span class="final-price">Rp {{ number_format($order->final_price, 0, ',', '.') }}</span>
                            @if($order->original_price > $order->final_price)
                                <span class="original-price">Rp {{ number_format($order->original_price, 0, ',', '.') }}</span>
                            @endif
                        </div>
                    </td>
                    <td>
                        <span class="status-badge status-{{ $order->status }}">
                            {{ $order->getStatusLabel() }}
                        </span>
                        @if($order->status === 'pending' && $order->isExpired())
                            <span class="expired-notice">Expired</span>
                        @endif
                    </td>
                    <td>
                        <div class="date-info">
                            <span class="date">{{ $order->created_at->format('d/m/Y') }}</span>
                            <span class="time">{{ $order->created_at->format('H:i') }}</span>
                        </div>
                    </td>
                    <td>
                        <div class="action-buttons">
                            <a href="{{ route('admin.orders.show', $order) }}" class="btn btn-sm btn-info" title="Detail">
                                <i class="fas fa-eye"></i>
                            </a>
                            <form action="{{ route('admin.orders.destroy', $order) }}" method="POST" class="inline-form" 
                                  onsubmit="return confirm('Yakin ingin menghapus pesanan ini?')">
                                @csrf
                                @method('DELETE')
                                <button type="submit" class="btn btn-sm btn-danger" title="Hapus">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </form>
                        </div>
                    </td>
                </tr>
                @empty
                <tr>
                    <td colspan="9" class="text-center">
                        <div class="empty-state">
                            <i class="fas fa-inbox"></i>
                            <p>Tidak ada pesanan ditemukan</p>
                        </div>
                    </td>
                </tr>
                @endforelse
            </tbody>
        </table>
    </div>
</form>

<!-- Pagination -->
<div class="pagination-wrapper">
    {{ $orders->appends(request()->query())->links() }}
</div>

<style>
.order-filters {
    background: var(--card-bg);
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
    border: 1px solid var(--border-color);
}

.filter-tabs {
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
    margin-bottom: 1rem;
}

.filter-tab {
    padding: 0.5rem 1rem;
    border-radius: 8px;
    background: var(--background);
    color: var(--text-secondary);
    text-decoration: none;
    font-size: 0.9rem;
    transition: all 0.2s;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.filter-tab:hover {
    background: var(--primary-color);
    color: white;
}

.filter-tab.active {
    background: var(--primary-color);
    color: white;
}

.filter-tab .badge {
    background: rgba(255,255,255,0.2);
    padding: 0.15rem 0.5rem;
    border-radius: 10px;
    font-size: 0.75rem;
}

.filter-tab .badge.warning { background: #f39c12; color: white; }
.filter-tab .badge.info { background: #3498db; color: white; }
.filter-tab .badge.success { background: #27ae60; color: white; }
.filter-tab .badge.danger { background: #e74c3c; color: white; }

.search-form {
    display: flex;
    gap: 0.5rem;
}

.search-input {
    flex: 1;
    padding: 0.75rem 1rem;
    border: 1px solid var(--border-color);
    border-radius: 8px;
    background: var(--background);
    color: var(--text-primary);
}

.bulk-actions {
    background: var(--primary-color);
    color: white;
    padding: 1rem;
    border-radius: 8px;
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 1rem;
}

.selected-count {
    font-weight: 500;
}

.bulk-action-select {
    padding: 0.5rem 1rem;
    border-radius: 6px;
    border: none;
    background: rgba(255,255,255,0.2);
    color: white;
}

.order-id {
    font-family: monospace;
    color: var(--primary-color);
}

.flash-badge {
    font-size: 0.8rem;
    margin-left: 0.25rem;
}

.product-info {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.product-name {
    font-weight: 500;
    color: var(--text-primary);
}

.game-name {
    font-size: 0.8rem;
    color: var(--text-secondary);
}

.price-info {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.final-price {
    font-weight: 600;
    color: var(--primary-color);
}

.original-price {
    font-size: 0.8rem;
    color: var(--text-secondary);
    text-decoration: line-through;
}

.status-badge {
    display: inline-block;
    padding: 0.35rem 0.75rem;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 500;
}

.status-pending { background: #fff3cd; color: #856404; }
.status-paid { background: #cce5ff; color: #004085; }
.status-processing { background: #d4edda; color: #155724; }
.status-completed { background: #28a745; color: white; }
.status-failed { background: #f8d7da; color: #721c24; }
.status-expired { background: #6c757d; color: white; }
.status-refunded { background: #e2e3e5; color: #383d41; }

.expired-notice {
    display: block;
    font-size: 0.7rem;
    color: #e74c3c;
    margin-top: 0.25rem;
}

.date-info {
    display: flex;
    flex-direction: column;
    gap: 0.15rem;
}

.date-info .date {
    font-weight: 500;
}

.date-info .time {
    font-size: 0.8rem;
    color: var(--text-secondary);
}

.action-buttons {
    display: flex;
    gap: 0.5rem;
}

.btn-sm {
    padding: 0.35rem 0.6rem;
    font-size: 0.8rem;
}

.inline-form {
    display: inline;
}

.checkbox {
    width: 18px;
    height: 18px;
    cursor: pointer;
}

.empty-state {
    padding: 3rem;
    color: var(--text-secondary);
}

.empty-state i {
    font-size: 3rem;
    margin-bottom: 1rem;
    opacity: 0.5;
}

.pagination-wrapper {
    margin-top: 1.5rem;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.order-checkbox');
    const bulkActions = document.querySelector('.bulk-actions');
    const selectedCount = document.querySelector('.selected-count');

    function updateBulkActions() {
        const checked = document.querySelectorAll('.order-checkbox:checked');
        if (checked.length > 0) {
            bulkActions.style.display = 'flex';
            selectedCount.textContent = checked.length + ' pesanan dipilih';
        } else {
            bulkActions.style.display = 'none';
        }
    }

    selectAll.addEventListener('change', function() {
        checkboxes.forEach(cb => cb.checked = this.checked);
        updateBulkActions();
    });

    checkboxes.forEach(cb => {
        cb.addEventListener('change', updateBulkActions);
    });
});
</script>
@endsection
