<?php

namespace App\Notifications;

use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class ResetPasswordNotification extends Notification
{
    /**
     * @var string
     */
    public $token;

    /**
     * @var string|null
     */
    public $phoneNumber;

    /**
     * @var string
     */
    public $sendMode;

    /**
     * Create a new notification instance.
     *
     * @param  string  $token
     * @param  string|null  $phoneNumber
     * @param  string  $sendMode
     * @return void
     */
    public function __construct($token, $phoneNumber = null, $sendMode = 'email')
    {
        $this->token = $token;
        $this->phoneNumber = $phoneNumber;
        $this->sendMode = $sendMode;  // 'email' or 'whatsapp'
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array<int, string>
     */
    public function via($notifiable): array
    {
        $channels = [];

        // Tentukan channel berdasarkan mode
        if ($this->sendMode === 'email') {
            $channels[] = 'mail';
        } elseif ($this->sendMode === 'whatsapp' && $this->phoneNumber) {
            $channels[] = 'whatsapp';
        }

        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable): MailMessage
    {
        $url = route('password.reset', [
            'token' => $this->token,
            'email' => $notifiable->getEmailForPasswordReset(),
        ]);

        return (new MailMessage)
            ->theme('webtopup')
            ->subject('Reset Password - ' . config('app.name'))
            ->greeting('Halo! 👋')
            ->line('Anda menerima email ini karena ada permintaan reset password untuk akun Anda di **' . config('app.name') . '**.')
            ->line('Klik tombol di bawah untuk melanjutkan proses reset password:')
            ->action('Reset Password Sekarang', $url)
            ->line('Link ini akan **kadaluarsa dalam ' . config('auth.passwords.' . config('auth.defaults.passwords') . '.expire') . ' menit**.')
            ->line('Jika Anda tidak meminta reset password, abaikan email ini dan akun Anda tetap aman.')
            ->line('---')
            ->line('**Tips Keamanan:** Jangan bagikan link ini kepada siapa pun.')
            ->salutation('Terima kasih,  
Team ' . config('app.name') . ' 🎮');
    }

    /**
     * Get the WhatsApp representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array{phone: string, message: string}
     */
    public function toWhatsApp($notifiable): array
    {
        $url = route('password.reset', [
            'token' => $this->token,
            'email' => $notifiable->getEmailForPasswordReset(),
        ]);

        return [
            'phone' => $this->phoneNumber,
            'message' => "Halo!\n\nAnda menerima pesan ini karena ada permintaan reset password untuk akun Anda di " . config('app.name') . ".\n\nKlik link berikut untuk reset password:\n{$url}\n\nLink akan kadaluarsa dalam " . config('auth.passwords.' . config('auth.defaults.passwords') . '.expire') . " menit.\n\nJika Anda tidak meminta reset password, abaikan pesan ini.\n\nTerima kasih,\nTim " . config('app.name')
        ];
    }
}
