<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class FlashSale extends Model
{
    use HasFactory;

    protected $fillable = [
        'product_id',
        'flash_price',
        'flash_stock',
        'sold',
        'start_time',
        'end_time',
        'is_active',
    ];

    protected $casts = [
        'flash_price' => 'decimal:2',
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'is_active' => 'boolean',
    ];

    /**
     * Get the product for this flash sale
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Check if flash sale is currently active (within time range and has stock)
     */
    public function isCurrentlyActive()
    {
        $now = Carbon::now();
        return $this->is_active &&
            $now->between($this->start_time, $this->end_time) &&
            $this->getRemainingStock() > 0;
    }

    /**
     * Check if flash sale has started
     */
    public function hasStarted()
    {
        return Carbon::now()->gte($this->start_time);
    }

    /**
     * Check if flash sale has ended
     */
    public function hasEnded()
    {
        return Carbon::now()->gt($this->end_time);
    }

    /**
     * Check if flash sale is upcoming (not started yet)
     */
    public function isUpcoming()
    {
        return Carbon::now()->lt($this->start_time);
    }

    /**
     * Get remaining stock for flash sale
     */
    public function getRemainingStock()
    {
        return max(0, $this->flash_stock - $this->sold);
    }

    /**
     * Check if stock is sold out
     */
    public function isSoldOut()
    {
        return $this->getRemainingStock() <= 0;
    }

    /**
     * Get discount percentage compared to original product price
     */
    public function getDiscountPercentage()
    {
        if (!$this->product) {
            return 0;
        }

        $originalPrice = $this->product->price;
        if ($originalPrice <= 0) {
            return 0;
        }

        return round((($originalPrice - $this->flash_price) / $originalPrice) * 100);
    }

    /**
     * Get time remaining in seconds
     */
    public function getTimeRemainingInSeconds()
    {
        if ($this->hasEnded()) {
            return 0;
        }

        if (!$this->hasStarted()) {
            return $this->start_time->diffInSeconds(Carbon::now());
        }

        return $this->end_time->diffInSeconds(Carbon::now());
    }

    /**
     * Get flash sale status label
     */
    public function getStatusLabel()
    {
        if (!$this->is_active) {
            return 'Nonaktif';
        }

        if ($this->isUpcoming()) {
            return 'Akan Datang';
        }

        if ($this->hasEnded()) {
            return 'Berakhir';
        }

        if ($this->isSoldOut()) {
            return 'Habis Terjual';
        }

        return 'Sedang Berlangsung';
    }

    /**
     * Get flash sale status class for styling
     */
    public function getStatusClass()
    {
        if (!$this->is_active) {
            return 'secondary';
        }

        if ($this->isUpcoming()) {
            return 'info';
        }

        if ($this->hasEnded() || $this->isSoldOut()) {
            return 'danger';
        }

        return 'success';
    }

    /**
     * Scope to get only active flash sales
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to get currently running flash sales
     */
    public function scopeCurrentlyRunning($query)
    {
        $now = Carbon::now();
        return $query
            ->where('is_active', true)
            ->where('start_time', '<=', $now)
            ->where('end_time', '>=', $now)
            ->whereRaw('flash_stock > sold');
    }

    /**
     * Scope to get upcoming flash sales
     */
    public function scopeUpcoming($query)
    {
        $now = Carbon::now();
        return $query
            ->where('is_active', true)
            ->where('start_time', '>', $now);
    }
}
